<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Api;

use Scalapay\Scalapay\Service\LoggerService;
use Scalapay\Scalapay\Service\SettingsService;
use Shopware\Core\Framework\Context;

class Connector
{
    /**
     * @var LoggerService
     */
    protected LoggerService $loggerService;

    /**
     * @var SettingsService
     */
    protected SettingsService $settingsService;

    /**
     * @var Client
     */
    protected Client $client;

    /**
     * @param LoggerService $loggerService
     * @param SettingsService $settingsService
     */
    public function __construct(LoggerService $loggerService, SettingsService $settingsService)
    {
        $this->loggerService = $loggerService;
        $this->settingsService = $settingsService;
        $apiConfigs = $this->getApiConfigs();
        $this->client = new Client($apiConfigs['endpoint'], $apiConfigs['key']);
    }

    /**
     * @param array $orderPayload
     * @return array
     */
    public function createOrder(
        array $orderPayload
    ): array {
        return $this->doRequest('/v2/orders', 'POST', $orderPayload);
    }


    /**
     * @param array $capturePayload
     * @return array
     */
    public function capture(
        array $capturePayload
    ): array {
        return $this->doRequest('/v2/payments/capture', 'POST', $capturePayload);
    }

    /**
     * @param string $token
     * @param array $refundPayload
     * @return array
     */
    public function refund(
        string $token,
        array $refundPayload
    ): array {
        return $this->doRequest("/v2/payments/$token/refund", 'POST', $refundPayload);
    }

    /**
     * @param string $token
     * @return array
     */
    public function getOrderDetails(
        string $token
    ): array {
        return $this->doRequest("/v2/payments/$token", 'GET');
    }

    /**
     * @param string $token
     * @param array $voidPayload
     * @return array
     */
    public function void(
        string $token,
        array $voidPayload
    ): array {
        return $this->doRequest("/v2/payments/$token/void", 'POST', $voidPayload);
    }

    /**
     * @param string $path
     * @param string $method
     * @param array|null $payload
     * @return array
     */
    private function doRequest(string $path, string $method, array $payload = null): array
    {
        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $this->client->getEndpoint() . $path);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Accept: application/json",
                "Authorization: Bearer " . $this->client->getApiKey(),
                "Content-Type: application/json"
            ]);

        if (!empty($payload) && 'POST' === strtoupper($method)) {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        }

        $rawResponse = curl_exec($ch);
        $response = [
            ...json_decode($rawResponse, true),
            'httpStatusCode' => (int)curl_getinfo($ch, CURLINFO_HTTP_CODE)
        ];

        $this->loggerService->addEntry(
            sprintf('[Api Connector] %s %s Data', $method, $path),
            Context::createDefaultContext(),
            null,
            [
                'request' => [
                    'url' => $this->client->getEndpoint() . $path,
                    'method' => $method,
                    'payload' => $payload
                ],
                'response' => $response
            ]
        );

        if (curl_errno($ch)) {
            $this->loggerService->addEntry(
                '[Api Connector] Curl Error Message: ' . curl_error($ch),
                Context::createDefaultContext()
            );
        }

        curl_close($ch);

        return $response;
    }

    /**
     * @return array
     */
    private function getApiConfigs(): array
    {
        $scalapayMode = $this->settingsService->getSetting('paymentScalapayMode');

        if (isset($scalapayMode) && !empty($scalapayMode) && 'live' === strtolower($scalapayMode)) {
            return [
                'key' => $this->settingsService->getSetting('paymentScalapayLiveEncyptionkey'),
                'endpoint' => $this->settingsService->getSetting('paymentScalapayProductionUrl')
            ];
        }

        return [
            'key' => $this->settingsService->getSetting('paymentScalapayEncyptionKey'),
            'endpoint' => $this->settingsService->getSetting('paymentScalapayTestUrl')
        ];
    }


}