<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Controller\Administration;

use Exception;
use Scalapay\Scalapay\Api\Connector;
use Scalapay\Scalapay\Helper\GatewayHelper;
use Scalapay\Scalapay\Helper\OrderStatusHelper;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

#[Route(defaults: ['_routeScope' => ['api']])]
class RefundController extends AbstractController
{

    /**
     * @var Connector
     */
    private Connector $connector;

    /**
     * @var GatewayHelper
     */
    private GatewayHelper $gatewayHelper;

    /**
     * @var EntityRepository
     */
    private EntityRepository $orderRepository;

    /**
     * @var OrderStatusHelper
     */
    private OrderStatusHelper $orderStatusHelper;

    /**
     * @param Connector $connector
     * @param GatewayHelper $gatewayHelper
     * @param EntityRepository $orderRepository
     * @param OrderStatusHelper $orderStatusHelper
     */
    public function __construct(
        Connector $connector,
        GatewayHelper $gatewayHelper,
        EntityRepository $orderRepository,
        OrderStatusHelper $orderStatusHelper
    ) {
        $this->connector = $connector;
        $this->gatewayHelper = $gatewayHelper;
        $this->orderRepository = $orderRepository;
        $this->orderStatusHelper = $orderStatusHelper;
    }


    #[Route(path: "/api/scalapay/get-refund-data", name: "api.action.scalapay.get-refund-data", defaults: ['_routeScope' => ['api']], methods: ['POST'])]
    public function getRefundData(Request $request, Context $context): JsonResponse
    {
        $orderId = $request->get('orderId');
        $orderToken = $request->get('orderToken');

        if (!$this->gatewayHelper->isScalapayPaymentMethod($orderId, $context)) {
            return new JsonResponse(['isAllowed' => false, 'refundedAmount' => 0]);
        }

        try {
            $apiResponse = $this->connector->getOrderDetails($orderToken);

            if (empty($apiResponse['token']) || !isset($apiResponse["totalAmount"]['amount'])) {
                throw new Exception('[SCALAPAY REFUND] GetOrderDetails Request Error');
            }

             return new JsonResponse(['isAllowed' => true, 'refundedAmount' => $apiResponse["totalAmount"]['amount']]);
        } catch (Exception) {
            return new JsonResponse(['isAllowed' => true, 'refundedAmount' => 0]);
        }
    }

    #[Route(path:"/api/scalapay/refund", name: "api.action.scalapay.refund", defaults: ['_routeScope' => ['api']], methods: ['POST'])]
    public function refund(Request $request, Context $context): JsonResponse
    {
        $amount = $request->get('amount');
        $merchantReference = $request->get('merchantReference');
        $orderToken = $request->get('orderToken');
        $refundedAmount = $request->get('refundedAmount');

        $order = $this->getOrder($merchantReference, $context);

        $refundPayload = array(
            'refundAmount' => array(
                'amount' => (string)$amount,
                'currency' => $order->getCurrency()->getIsoCode()
            ),
            'merchantReference' => (string)$merchantReference
        );

        try {
            $apiResponse = $this->connector->refund($orderToken, $refundPayload);
            if (200 !== $apiResponse['httpStatusCode'] || empty($apiResponse['token'] ?? $apiResponse['refundToken'])) {
                return new JsonResponse([
                    'status' => false,
                    'message' => $apiResponse['message'] ?? 'Unknown Refund Error'
                ]);
            }
            $refundStatus = ((float)$refundedAmount + (float)$amount === $order->getAmountTotal()) ? 'refunded' : 'partial_refunded';
            $this->orderStatusHelper->transitionPaymentState($refundStatus, $order->getTransactions()->first()->getId(), $context);

            return new JsonResponse([
                'status' => true,
                'message' => 'Refund successful'
            ]);
        } catch (Exception $e) {
            return new JsonResponse([
                'status' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * @param string $orderId
     * @param Context $context
     * @return OrderEntity
     * @throws InconsistentCriteriaIdsException
     */
    private function getOrder(string $orderId, Context $context): OrderEntity
    {
        $criteria = new Criteria([$orderId]);
        $criteria->addAssociation('currency');
        $criteria->addAssociation('transactions.paymentMethod');
        $criteria->addAssociation('transactions.paymentMethod.plugin');

        /** @var OrderEntity $order */
        return $this->orderRepository->search($criteria, $context)->get($orderId);
    }
}
