<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Handlers;

use Exception;
use Scalapay\Scalapay\Api\Connector;
use Scalapay\Scalapay\Helper\CheckoutHelper;
use Scalapay\Scalapay\Helper\GatewayHelper;
use Scalapay\Scalapay\Helper\OrderStatusHelper;
use Scalapay\Scalapay\Service\LoggerService;
use Scalapay\Scalapay\Service\SettingsService;
use Shopware\Core\Checkout\Cart\CartException;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStateHandler;
use Shopware\Core\Checkout\Payment\Cart\AsyncPaymentTransactionStruct;
use Shopware\Core\Checkout\Payment\Cart\PaymentHandler\AsynchronousPaymentHandlerInterface;
use Shopware\Core\Checkout\Payment\Exception\AsyncPaymentFinalizeException;
use Shopware\Core\Checkout\Payment\Exception\AsyncPaymentProcessException;
use Shopware\Core\Checkout\Payment\PaymentException;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\StateMachine\Exception\IllegalTransitionException;
use Shopware\Core\System\StateMachine\Exception\StateMachineNotFoundException;
use Shopware\Core\System\StateMachine\Exception\StateMachineStateNotFoundException;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\RouterInterface;

class ScalapayPaymentHandler implements AsynchronousPaymentHandlerInterface
{
    /**
     * @var CheckoutHelper
     */
    protected CheckoutHelper $checkoutHelper;

    /**
     * @var Connector
     */
    protected Connector $connector;

    /**
     * @var string
     */
    protected string $environment;

    /**
     * @var GatewayHelper
     */
    protected GatewayHelper $gatewayHelper;

    /**
     * @var LoggerService
     */
    protected LoggerService $loggerService;

    /**
     * @var OrderStatusHelper
     */
    protected OrderStatusHelper $orderStatusHelper;

    /**
     * @var EntityRepository
     */
    protected EntityRepository $orderTransactionRepository;

    /**
     * @var OrderTransactionStateHandler
     */
    protected OrderTransactionStateHandler $orderTransactionStateHandler;

    /**
     * @var RouterInterface
     */
    protected RouterInterface $router;

    /**
     * @var SettingsService
     */
    protected SettingsService $settingsService;

    /**
     * @param CheckoutHelper $checkoutHelper
     * @param Connector $connector
     * @param string $environment
     * @param GatewayHelper $gatewayHelper
     * @param LoggerService $loggerService
     * @param OrderStatusHelper $orderStatusHelper
     * @param EntityRepository $orderTransactionRepository
     * @param OrderTransactionStateHandler $orderTransactionStateHandler
     * @param RouterInterface $router
     * @param SettingsService $settingsService
     */
    public function __construct(
        CheckoutHelper $checkoutHelper,
        Connector $connector,
        string $environment,
        GatewayHelper $gatewayHelper,
        LoggerService $loggerService,
        OrderStatusHelper $orderStatusHelper,
        EntityRepository $orderTransactionRepository,
        OrderTransactionStateHandler $orderTransactionStateHandler,
        RouterInterface $router,
        SettingsService $settingsService
    ) {
        $this->checkoutHelper = $checkoutHelper;
        $this->connector = $connector;
        $this->environment = $environment;
        $this->gatewayHelper = $gatewayHelper;
        $this->loggerService = $loggerService;
        $this->orderStatusHelper = $orderStatusHelper;
        $this->orderTransactionRepository = $orderTransactionRepository;
        $this->orderTransactionStateHandler = $orderTransactionStateHandler;
        $this->router = $router;
        $this->settingsService = $settingsService;
    }


    /**
     * @param AsyncPaymentTransactionStruct $transaction
     * @param RequestDataBag $dataBag
     * @param SalesChannelContext $salesChannelContext
     * @param string|null $gateway
     * @param string $type
     * @param array $gatewayInfo
     * @return RedirectResponse
     * @throws AsyncPaymentProcessException
     */
    public function pay(
        AsyncPaymentTransactionStruct $transaction,
        RequestDataBag $dataBag,
        SalesChannelContext $salesChannelContext,
        string $gateway = null,
        string $type = 'redirect',
        array $gatewayInfo = []
    ): RedirectResponse {
        try {
            $order = $this->checkoutHelper->getOrderWithAssociations(
                    $transaction->getOrder()->getId(),
                    $salesChannelContext->getContext()
                )
                ?? $transaction->getOrder();
            $transactionId = $transaction->getOrderTransaction()->getId();
            if (null === $order) {
                throw new Exception('Cant find order in pay method');
            }

            $customer = $order->getOrderCustomer()?->getCustomer() ?? $salesChannelContext->getCustomer();
            if (null === $customer) {
                throw new Exception(CartException::customerNotLoggedIn()->getMessage());
            }

            $orderData = $this->checkoutHelper->prepareScalapayOrderData(
                $order,
                $customer,
                $transaction,
                $salesChannelContext
            );
            $productTypeDetails = $this->checkoutHelper->getProductTypeDetails();

            $this->loggerService->addEntry('Scalapay Order Data', $salesChannelContext->getContext(), null, [
                'salesChannelId' => $transaction->getOrder()->getSalesChannelId(),
                'transactionId' => $transactionId,
                'environment' => $this->environment,
                'order ' => $order,
                'customer' => $customer,
                'request' =>  $this->gatewayHelper->getGlobals(),
                'apiPayload' => $orderData,
                'product' => $productTypeDetails,
            ]);

            $orderResponse = $this->connector->createOrder(array_merge($orderData, $productTypeDetails));
            if (200 !== $orderResponse['httpStatusCode'] || !isset($orderResponse['token']) || !isset($orderResponse['checkoutUrl'])) {
                throw new Exception(
                    isset($orderResponse['message']) ? (
                    is_array($orderResponse['message']) ? json_encode(
                        $orderResponse['message']
                    ) : $orderResponse['message'])
                        : 'Unknown Error in POST /orders API request'
                );
            }

            return new RedirectResponse($orderResponse["checkoutUrl"]);
        } catch (Exception $e) {
            $errorMessage = sprintf(
                '[Order ID %s] Error while processing payment: message -> %s ',
                $transaction->getOrder()->getId(),
                $e->getMessage()
            );
            $this->loggerService->addEntry(
                $errorMessage,
                $salesChannelContext->getContext(),
                $e,
                [
                    'function' => 'pay',
                ]
            );

            throw new AsyncPaymentProcessException(
                $transaction->getOrderTransaction()->getId(),
                $errorMessage
            );
        }
    }

    /**
     * @param AsyncPaymentTransactionStruct $transaction
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return void
     * @throws CustomerCanceledAsyncPaymentException
     * @throws AsyncPaymentFinalizeException
     */
    public function finalize(
        AsyncPaymentTransactionStruct $transaction,
        Request $request,
        SalesChannelContext $salesChannelContext
    ): void {
        $transactionId = $transaction->getOrderTransaction()->getId();
        $orderId = $transaction->getOrder()->getOrderNumber();
        $orderAmount = $transaction->getOrder()->getAmountTotal();
        $context = $salesChannelContext->getContext();

        $requestOrderId = $request->query->get('orderId');
        $requestOrderAmount = $request->query->get('amount');
        $requestOrderToken = $request->query->get('orderToken');
        $requestStatus = $request->query->get('status');

        if ($request->query->getBoolean('cancel')) {
            $this->loggerService->addEntry(
                sprintf('[Order ID %s - Token %s] Cancel param received in finalize payment flow, exiting.',
                    $orderId,
                    $requestOrderToken
                ),
                $salesChannelContext->getContext(),
                null,
                [
                    'function' => 'finalize',
                ]
            );

            throw PaymentException::customerCanceled(
                $transactionId,
                sprintf(
                    '[Order ID %s - Token %s] Cancel param returned.',
                    $orderId,
                    $requestOrderToken
                )
            );
        }

        try {
            if ('success' !== strtolower($requestStatus)) {
                throw new Exception(
                    sprintf(
                        '[Order ID %s - Token %s] Request cancelled by user on scalapay / Some errors occurred.',
                        $orderId,
                        $requestOrderToken
                    )
                );
            }

            if (empty($requestOrderToken)) {
                throw new Exception(
                    sprintf(
                        '[Order ID %s - Token %s] Missing orderToken in response.',
                        $orderId,
                        $requestOrderToken
                    )
                );
            }

            if ((float)$requestOrderAmount !== $orderAmount) {
                throw new Exception(
                    sprintf(
                        '[Order ID %s - Token %s] Amount mismatch between session order total (%s) and scalapay returned total (%s).',
                        $orderId,
                        $requestOrderToken,
                        $orderAmount,
                        $requestOrderAmount
                    )
                );
            }

            if ($orderId !== $requestOrderId) {
                throw new Exception(
                    sprintf(
                        '[Order ID %s - Token %s] OrderId mismatch between session id (%s) and scalapay returned id (%s).',
                        $orderId,
                        $requestOrderToken,
                        $orderId,
                        $requestOrderId
                    )
                );
            }

            $storedTransactionCustomFields = $transaction->getOrderTransaction()->getCustomFields() ?: [];
            if (!empty($storedTransactionCustomFields['scalapay_order_token'])) {
                // redirect to thank you page
                return;
            }

            $updatedCustomFields = array_merge(
                $storedTransactionCustomFields,
                ['scalapay_order_token' => $requestOrderToken]
            );
            $transaction->getOrderTransaction()->setCustomFields($updatedCustomFields);
            $context->scope(
                Context::SYSTEM_SCOPE,
                function (Context $context) use ($transactionId, $updatedCustomFields) {
                    $this->orderTransactionRepository->update([
                        [
                            'id' => $transactionId,
                            'customFields' => $updatedCustomFields,
                        ]
                    ], $context);
                }
            );

            $captureResponse = $this->connector->capture(
                ['token' => $requestOrderToken, 'merchantReference' => $orderId]
            );

            if (200 !== $captureResponse['httpStatusCode'] && !isset($captureResponse['status']) || 'approved' !== strtolower($captureResponse['status'])) {
                $this->connector->void($requestOrderToken, [
                    'merchantReference' => $orderId,
                ]);

                $this->orderTransactionStateHandler->reopen(
                    $transactionId,
                    $salesChannelContext->getContext()
                );

                throw new Exception(
                    sprintf(
                        '[Order ID %s - Token %s] Capture request failed.',
                        $orderId,
                        $requestOrderToken
                    )
                );
            }

            try {
                $this->orderTransactionStateHandler->paid(
                    $transactionId,
                    $salesChannelContext->getContext()
                );

                $configOrderState = $this->settingsService->getSetting(
                    'orderStateWithAPaidTransaction',
                    $salesChannelContext->getSalesChannel()->getId()
                );

                $fullOrder = $this->checkoutHelper->getOrderWithAssociations(
                    $transaction->getOrder()->getId(),
                    $salesChannelContext->getContext()
                );

                if ($fullOrder === null) {
                    throw new \Exception(sprintf('Cannot load full order entity for ID: %s', $orderId));
                }

                $this->orderStatusHelper->setOrderState(
                    $fullOrder,
                    $configOrderState,
                    $salesChannelContext->getContext()
                );

            } catch (InconsistentCriteriaIdsException|IllegalTransitionException|StateMachineNotFoundException
            |StateMachineStateNotFoundException $e) {
                $this->orderTransactionStateHandler->reopen(
                    $transactionId,
                    $salesChannelContext->getContext()
                );
                throw new Exception($e->getMessage());
            }
        } catch (Exception $e) {
            $this->loggerService->addEntry(
                'Error in finalize payment flow -> ' . $e->getMessage(),
                $salesChannelContext->getContext(),
                $e,
                [
                    'function' => 'finalize',
                ]
            );
            throw new AsyncPaymentFinalizeException($transactionId, $e->getMessage());
        }
    }
}
