<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Helper;

use Exception;
use Shopware\Core\Checkout\Customer\CustomerEntity;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Checkout\Payment\Cart\AsyncPaymentTransactionStruct;
use Shopware\Core\Checkout\Promotion\Cart\PromotionProcessor;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Plugin\Exception\PluginNotFoundException;
use Shopware\Core\Framework\Plugin\PluginService;

class CheckoutHelper
{
    /**
     * @var EntityRepository
     */
    protected EntityRepository $orderRepository;

    /**
     * @var PluginService
     */
    private PluginService $pluginService;

    /**
     * @var string
     */
    private string $shopwareVersion;

    /**
     * @param EntityRepository $orderRepository
     * @param PluginService $pluginService
     * @param string $shopwareVersion
     */
    public function __construct(
        EntityRepository $orderRepository,
        PluginService $pluginService,
        string $shopwareVersion
    ) {
        $this->orderRepository = $orderRepository;
        $this->pluginService = $pluginService;
        $this->shopwareVersion = $shopwareVersion;
    }

    /**
     * @param $order
     * @param $customer
     * @param $transaction
     * @param $salesChannelContext
     * @return array
     */
    public function prepareScalapayOrderData($order, $customer, $transaction, $salesChannelContext): array
    {
        $orderData = [];
        $orderData['totalAmount'] = $this->getAmounts($order, 'order');
        $orderData['shippingAmount'] = $this->getAmounts($order, 'shipping');
        $orderData['taxAmount'] = $this->getAmounts($order, 'tax');

        $discountData = $this->getDiscountTotal($order);
        if (count($discountData)) {
            $orderData['discounts'] = $discountData;
        }

        $orderData['consumer'] = $this->getCustomerData($customer);
        $orderData['billing'] = $this->getAddressData($customer);
        $orderData['shipping'] = $this->getAddressData($customer, 'shipping');
        $orderData['items'] = $this->getShoppingCart($order);
        $orderData['merchant'] = $this->getPaymentOptions($transaction);
        $orderData['extensions']['pluginDetails'] = $this->getPluginMetadata($salesChannelContext->getContext());

        return $orderData;
    }

    /**
     * @param string $orderId
     * @param Context $context
     * @return OrderEntity|null
     */
    public function getOrderWithAssociations(string $orderId, Context $context): ?OrderEntity
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('id', $orderId));
            $criteria->addAssociation('currency');
            $criteria->addAssociation('addresses');
            $criteria->addAssociation('language');
            $criteria->addAssociation('language.locale');
            $criteria->addAssociation('lineItems');
            $criteria->addAssociation('lineItems.product');
            $criteria->addAssociation('lineItems.product.media');
            $criteria->addAssociation('deliveries');
            $criteria->addAssociation('deliveries.shippingOrderAddress');
            $criteria->addAssociation('transactions');
            $criteria->addAssociation('transactions.paymentMethod');
            $criteria->addAssociation('stateMachineState');

            return $this->orderRepository->search($criteria, $context)->first();
        } catch (Exception) {
            return null;
        }
    }

    /**
     *
     * @return array
     */
    public function getProductTypeDetails():array {
        return [
            'type'      => 'online',
        ];
    }

    /**
     * @param Context $context
     * @return array
     * @throws PluginNotFoundException
     */
    private function getPluginMetadata(Context $context): array
    {
        return [
            'platform' => 'Shopware',
            'customized' => '0',
            'platformVersion' => $this->shopwareVersion,
            'pluginVersion' => $this->pluginService->getPluginByName('ScalapayPayments', $context)->getVersion() // todo fix without using internal class?
        ];
    }

    /**
     * @param CustomerEntity $customer
     * @return array
     */
    private function getCustomerData(CustomerEntity $customer): array
    {
        if (!empty($firstName = $customer->getDefaultBillingAddress()->getFirstName())) {
            $consumerInfo['givenNames'] = $firstName;
        }

        if (!empty($lastName = $customer->getDefaultBillingAddress()->getLastName())) {
            $consumerInfo['surname'] = $lastName;
        }

        if (!empty($phone = $customer->getDefaultBillingAddress()->getPhoneNumber())) {
            $consumerInfo['phoneNumber'] = $phone;
        }

        if (!empty($email = $customer->getEmail())) {
            $consumerInfo['email'] = $email;
        }

        return $consumerInfo ?? [];
    }

    /**
     * @param CustomerEntity $customer
     * @param string $type
     * @return array
     */
    private function getAddressData(CustomerEntity $customer, string $type = 'billing'): array
    {
        $address = 'billing' === $type ? $customer->getDefaultBillingAddress() : $customer->getDefaultShippingAddress();

        if (!empty($name = $address->getFirstName())) {
            $addressInfo['name'] = $name . ' ' . $address->getLastName();
        }

        if (!empty($street = $address->getStreet())) {
            $addressInfo['line1'] = $street;
        }

        if (!empty($zip = $address->getZipcode())) {
            $addressInfo['postcode'] = $zip;
        }

        if (!empty($state = $address->getCountryState()?->getName())) {
            $addressInfo['state'] = $state;
        }

        if (!empty($city = $address->getCity())) {
            $addressInfo['suburb'] = $city;
        }

        if (!empty($country = $address->getCountry()?->getIso())) {
            $addressInfo['countryCode'] = $country;
        }

        if (!empty($phone = $address->getPhoneNumber())) {
            $addressInfo['phoneNumber'] = $phone;
        }

        return $addressInfo ?? [];
    }

    /**
     * @param AsyncPaymentTransactionStruct $transaction
     * @return array
     */
    private function getPaymentOptions(AsyncPaymentTransactionStruct $transaction): array
    {
        return [
            'redirectConfirmUrl' => sprintf('%s&orderId=%s&amount=%s',
                $transaction->getReturnUrl(), $transaction->getOrder()->getOrderNumber(), $transaction->getOrder()->getAmountTotal()),
            'redirectCancelUrl' => sprintf('%s&cancel=1', $transaction->getReturnUrl()),
        ];
    }

    /**
     * @param OrderEntity $order
     * @return array
     */
    private function getShoppingCart(OrderEntity $order): array
    {
        $lineItems = $order->getNestedLineItems();
        if (null === $lineItems || !$lineItems->count()) {
            return [];
        }

        foreach ($lineItems as $item) {
            if (in_array($item->getType(), [PromotionProcessor::LINE_ITEM_TYPE, 'customized-products']) || $item->getTotalPrice() < 0) {
                continue;
            }
            $orderItem["name"] = (string)$item->getLabel();
            $orderItem["sku"] = (string)$item->getProduct()?->getProductNumber() ?? '';
            $orderItem["quantity"] = (string)$item->getQuantity();
            $orderItem["price"]["amount"] = (string)$item->getPrice()->getUnitPrice();
            $orderItem["price"]["currency"] = $order->getCurrency()?->getIsoCode() ?? 'EUR';

            $shoppingCart[] = $orderItem;
        }

        return $shoppingCart ?? [];
    }

    /**
     * @param OrderEntity $order
     * @param string $type
     * @return string[]
     */
    private function getAmounts(OrderEntity $order, string $type): array
    {
        $amount = match ($type) {
            'order' =>  $order->getAmountTotal(),
            'tax' => $order->getShippingCosts()?->getCalculatedTaxes()->getAmount(),
            'shipping' => $order->getShippingCosts()?->getTotalPrice(),
        };

        return [
            'amount' => (string)$amount ?: '0',
            'currency' => (string)$order->getCurrency()?->getIsoCode() ?? 'EUR'
        ];
    }

    /**
     * @param OrderEntity $order
     * @return array
     */
    private function getDiscountTotal(OrderEntity $order): array
    {
        $discountsItem = [];
        $lineItems = $order->getNestedLineItems();

        if (null === $lineItems || !$lineItems->count()) {
            return [];
        }

        foreach ($lineItems as $item) {
            if ($item->getType() === PromotionProcessor::LINE_ITEM_TYPE ||
                $item->getTotalPrice() < 0) {
                $scalapay_discount_item = array();
                $scalapay_discount_item["displayName"] = (string)$item->getLabel();
                $scalapay_discount_item["amount"]["amount"] = (string)abs($item->getPrice()->getTotalPrice());
                $scalapay_discount_item["amount"]["currency"] = (string)$order->getCurrency()?->getIsoCode() ?? 'EUR';

                $discountsItem[] = $scalapay_discount_item;
            }
        }

        return $discountsItem;
    }
}
