<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);


namespace Scalapay\Scalapay\Helper;

use Scalapay\Scalapay\PaymentMethods\ScalapayPaymentMethod;
use Scalapay\Scalapay\ScalapayPayments;
use Scalapay\Scalapay\Service\SettingsService;
use Shopware\Core\Checkout\Cart\Cart;
use Shopware\Core\Checkout\Cart\SalesChannel\CartService;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\HttpFoundation\Request;

class GatewayHelper
{
    public const SCALAPAY_GATEWAYS = [
        ScalapayPaymentMethod::class,
    ];

    /**
     * @var CartService;
     */
    private CartService $cartService;

    /**
     * @var EntityRepository
     */
    private EntityRepository $orderRepository;

    /**
     * @var SettingsService
     */
    private SettingsService $settingsService;

    /**
     * @param CartService $cartService
     * @param EntityRepository $orderRepository
     * @param SettingsService $settingsService
     */
    public function __construct(
        CartService $cartService,
        EntityRepository $orderRepository,
        SettingsService $settingsService
    ) {
        $this->cartService = $cartService;
        $this->orderRepository = $orderRepository;
        $this->settingsService = $settingsService;
    }

    /**
     * Check Scalapay restriction settings
     *
     * @param SalesChannelContext $context
     * @param ?Cart $cart
     * @param ?OrderEntity $order
     * @return bool
     */
    public function isScalapayAvailable(SalesChannelContext $context, ?Cart $cart = null,  ?OrderEntity $order = null): bool
    {
        if ('enabled' !== $this->settingsService->getSetting('paymentScalapayStatus')) {
            return false;
        }

        $cart = $cart ?: $this->cartService->getCart($context->getToken(), $context);
        $checkOrder = null !== $order && $order->getAmountTotal();

        // total amount check
        $totalAmount = $checkOrder ? $order->getAmountTotal() : $cart->getPrice()?->getTotalPrice();
        $hasItems = $checkOrder || (bool)$cart->getLineItems()?->count();
        if ($hasItems && (($totalAmount ?? 0) < $this->settingsService->getSetting(
                'paymentScalapayMinAmount'
            ) || $totalAmount > $this->settingsService->getSetting('paymentScalapayMaxAmount'))) {
            return false;
        }

        // shipping country check
        $shippingCountries = $checkOrder ? [$order->getDeliveries()?->getShippingAddress()?->first()?->getCountryId()] :
            $cart->getDeliveries()?->getAddresses()?->getCountryIds();

        if (!empty($shippingCountries ?? []) && !(bool)count(
            array_intersect($shippingCountries,
                $this->settingsService->getSetting('paymentScalapayCountries')
            )
        )) {
            return false;
        }

        // currencies check
        if (!in_array(
            $context->getCurrencyId(),
            $this->settingsService->getSetting('paymentScalapayCurrencies') ?? array()
        )) {
            return false;
        }

        // languages check
        if (!in_array(
            $context->getLanguageId(),
            $this->settingsService->getSetting('paymentScalapayLanguages') ?? array()
        )) {
            return false;
        }

        return true;
    }

    /**
     * @param string $orderId
     * @param Context $context
     * @return bool
     */
    public function isScalapayPaymentMethod(string $orderId, Context $context): bool
    {
        $transaction = $this->getOrderData($orderId, $context)?->getTransactions()->first();
        if (!$transaction || !$transaction->getPaymentMethod() || !$transaction->getPaymentMethod()->getPlugin()) {
            return false;
        }

        return $transaction->getPaymentMethod()->getPlugin()->getBaseClass() === ScalapayPayments::class;
    }

    /**
     * Retrieve super globals (replaces Request::createFromGlobals)
     *
     * @return Request
     */
    public function getGlobals(): Request
    {
        return new Request($_GET, $_POST, array(), $_COOKIE, $_FILES, $_SERVER);
    }

    /**
     * @param string $orderId
     * @param Context $context
     * @return mixed|null
     * @throws InconsistentCriteriaIdsException
     */
    private function getOrderData(string $orderId, Context $context): mixed
    {
        $criteria = new Criteria([$orderId]);
        $criteria->addAssociation('transactions');
        $criteria->addAssociation('transactions.paymentMethod');
        $criteria->addAssociation('transactions.paymentMethod.plugin');
        $criteria->addAssociation('salesChannel');

        return $this->orderRepository->search($criteria, $context)->get($orderId);
    }
}
