<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Helper;

use Exception;
use Scalapay\Scalapay\Service\LoggerService;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionEntity;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStateHandler;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionStates;
use Shopware\Core\Checkout\Order\OrderDefinition;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Checkout\Order\OrderStates;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateEntity;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineTransition\StateMachineTransitionActions;
use Shopware\Core\System\StateMachine\Exception\IllegalTransitionException;
use Shopware\Core\System\StateMachine\Exception\StateMachineInvalidEntityIdException;
use Shopware\Core\System\StateMachine\Exception\StateMachineInvalidStateFieldException;
use Shopware\Core\System\StateMachine\Exception\StateMachineNotFoundException;
use Shopware\Core\System\StateMachine\Exception\StateMachineStateNotFoundException;
use Shopware\Core\System\StateMachine\StateMachineRegistry;
use Shopware\Core\System\StateMachine\Transition;

class OrderStatusHelper
{
    /**
     * @var OrderTransactionStateHandler
     */
    private OrderTransactionStateHandler $orderTransactionStateHandler;

    /**
     * @var EntityRepository
     */
    private EntityRepository $transactionRepository;

    /**
     * @var EntityRepository
     */
    private EntityRepository $stateMachineRepository;

    /**
     * @var StateMachineRegistry
     */
    private StateMachineRegistry $stateMachineRegistry;

    /**
     * @var LoggerService
     */
    private LoggerService $loggerService;


    /**
     * @param OrderTransactionStateHandler $orderTransactionStateHandler
     * @param StateMachineRegistry $stateMachineRegistry
     * @param LoggerService $loggerService
     * @param EntityRepository $transactionRepository
     * @param EntityRepository $stateMachineRepository
     */
    public function __construct(
        OrderTransactionStateHandler $orderTransactionStateHandler,
        StateMachineRegistry $stateMachineRegistry,
        LoggerService $loggerService,
        EntityRepository $transactionRepository,
        EntityRepository $stateMachineRepository,
    ) {
        $this->loggerService = $loggerService;
        $this->stateMachineRegistry = $stateMachineRegistry;
        $this->transactionRepository = $transactionRepository;
        $this->orderTransactionStateHandler = $orderTransactionStateHandler;
        $this->stateMachineRepository = $stateMachineRepository;
    }

    /**
     * @param OrderEntity $order
     * @param string $orderState
     * @param Context $context
     * @return bool
     */
    public function setOrderState(OrderEntity $order, string $orderState, Context $context): bool
    {
        $currentStatus = $order->getStateMachineState()->getTechnicalName();

        // if current state is same as status that should be set, we don't need to do a transition
        if ($currentStatus === $orderState) {
            return false;
        }

        // Collect an array of possible order states
        $orderStates = [
            OrderStates::STATE_OPEN,
            OrderStates::STATE_IN_PROGRESS,
            OrderStates::STATE_COMPLETED,
            OrderStates::STATE_CANCELLED,
        ];

        // Check if the order state is valid
        if (!in_array($orderState, $orderStates, true)) {
            return false;
        }

        // Get the transition name
        $transitionName = match ($orderState) {
            OrderStates::STATE_OPEN => StateMachineTransitionActions::ACTION_REOPEN,
            OrderStates::STATE_IN_PROGRESS => StateMachineTransitionActions::ACTION_PROCESS,
            OrderStates::STATE_COMPLETED => StateMachineTransitionActions::ACTION_COMPLETE,
            OrderStates::STATE_CANCELLED => StateMachineTransitionActions::ACTION_CANCEL,
            default => null,
        };

        if (!$transitionName) {
            return false;
        }

        // Transition the order
        try {
            $this->stateMachineRegistry->transition(
                new Transition(
                    OrderDefinition::ENTITY_NAME,
                    $order->getId(),
                    $transitionName,
                    'stateId'
                ),
                $context
            );
        } catch (Exception $e) {
            $this->loggerService->addEntry(
                $e->getMessage(),
                $context,
                $e,
                [
                    'function' => 'setOrderState',
                ]
            );
            return false;
        }

        return true;
    }


    /**
     * @param string $status
     * @param string $orderTransactionId
     * @param Context $context
     * @throws IllegalTransitionException
     * @throws InconsistentCriteriaIdsException
     * @throws StateMachineInvalidEntityIdException
     * @throws StateMachineInvalidStateFieldException
     * @throws StateMachineNotFoundException
     * @throws StateMachineStateNotFoundException
     */
    public function transitionPaymentState(string $status, string $orderTransactionId, Context $context): void
    {
        $transitionAction = $this->getCorrectTransitionAction($status);

        if ($transitionAction === null) {
            return;
        }

        /**
         * Check if the state from the current transaction is equal
         * to the transaction we want to transition to.
         */
        if ($this->isSameStateId($transitionAction, $orderTransactionId, $context)) {
            return;
        }

        try {
            $functionName = $this->convertToFunctionName($transitionAction);
            $this->orderTransactionStateHandler->$functionName($orderTransactionId, $context);
        } catch (IllegalTransitionException) {
            if ($transitionAction !== StateMachineTransitionActions::ACTION_PAID) {
                return;
            }

            $this->orderTransactionStateHandler->reopen($orderTransactionId, $context);
            $this->transitionPaymentState($status, $orderTransactionId, $context);
        }
    }

    /**
     * @param string $status
     * @return string|null
     */
    private function getCorrectTransitionAction(string $status): ?string
    {
        return match ($status) {
            'completed' => StateMachineTransitionActions::ACTION_PAID,
            'declined', 'cancelled', 'void', 'expired' => StateMachineTransitionActions::ACTION_CANCEL,
            'refunded' => StateMachineTransitionActions::ACTION_REFUND,
            'partial_refunded' => StateMachineTransitionActions::ACTION_REFUND_PARTIALLY,
            'initialized' => StateMachineTransitionActions::ACTION_REOPEN,
            default => null,
        };
    }

    /**
     * @param string $transactionId
     * @param Context $context
     * @return OrderTransactionEntity
     * @throws InconsistentCriteriaIdsException
     */
    private function getTransaction(string $transactionId, Context $context): OrderTransactionEntity
    {
        $criteria = new Criteria([$transactionId]);
        /** @var OrderTransactionEntity $transaction */
        return $this->transactionRepository->search($criteria, $context)
            ->get($transactionId);
    }

    /**
     * @param string $actionName
     * @param string $orderTransactionId
     * @param Context $context
     * @return bool
     * @throws InconsistentCriteriaIdsException
     */
    private function isSameStateId(string $actionName, string $orderTransactionId, Context $context): bool
    {
        $transaction = $this->getTransaction($orderTransactionId, $context);
        $currentStateId = $transaction->getStateId();

        $actionStatusTransition = $this->getTransitionFromActionName($actionName, $context);
        $actionStatusTransitionId = $actionStatusTransition->getId();

        return $currentStateId === $actionStatusTransitionId;
    }

    /**
     * @param string $actionName
     * @param Context $context
     * @return StateMachineStateEntity
     * @throws InconsistentCriteriaIdsException
     */
    private function getTransitionFromActionName(string $actionName, Context $context): StateMachineStateEntity
    {
        $stateName = $this->getOrderTransactionStatesNameFromAction($actionName);
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('technicalName', $stateName));
        return $this->stateMachineRepository->search($criteria, $context)->first();
    }

    /**
     * @param string $actionName
     * @return string
     */
    private function getOrderTransactionStatesNameFromAction(string $actionName): string
    {
        return match ($actionName) {
            StateMachineTransitionActions::ACTION_PAID => OrderTransactionStates::STATE_PAID,
            StateMachineTransitionActions::ACTION_CANCEL => OrderTransactionStates::STATE_CANCELLED,
            default => OrderTransactionStates::STATE_OPEN,
        };
    }

    /**
     * Convert from snake_case to CamelCase.
     *
     * @param string $string
     * @return string
     */
    private function convertToFunctionName(string $string): string
    {
        $string = str_replace(' ', '', ucwords(str_replace('_', ' ', $string)));
        return lcfirst($string);
    }
}
