import './scalapay-refund.scss';
import template from './scalapay-refund.html.twig';

const {Mixin, Context} = Shopware;
const {Criteria} = Shopware.Data;

const scalapayFormattedHandlerIdentifier = 'handler_scalapay_scalapaypaymenthandler';

Shopware.Component.register('scalapay-refund', {
    template,
    inject: ['repositoryFactory', 'systemConfigApiService', 'scalapayApiService'],
    mixins: [Mixin.getByName('notification')],
    props: {
        orderId: {
            type: String, required: true
        },
    },
    data() {
        return {
            amount: null,
            isLoading: false,
            versionContext: null,
            order: null,
            maxRefundableAmount: 0,
            isRefundAllowed: true,
            refundedAmount: 0,
            showModal: false,
            isRefundDisabled: false,
            scalapayOrderToken: null,
            config: {}
        };
    },
    watch: {
        orderId() {
            this.createdComponent();
        }, amount() {
            this.amount = parseFloat(this.amount).toFixed(2);
        }, refundedAmount() {
            this.refundedAmount = parseFloat(this.refundedAmount).toFixed(2);
        }
    },
    methods: {
        closeModal() {
            this.showModal = false;
        }, showRefundModal() {
            if (this.amount < 0.01) {
                this.createNotificationWarning({
                    title: this.$tc('scalapay-payments.refund.notification.warningTitle'),
                    message: this.$tc('scalapay-payments.refund.notification.warningMessage')
                });
                return;
            }
            this.showModal = true;
        }, applyRefund() {
            this.closeModal()
            this.scalapayApiService.refund(this.amount, this.orderId, this.scalapayOrderToken, this.refundedAmount).then((response) => {

                if (response.status === false) {
                    this.createNotificationError({
                        title: this.$tc('scalapay-payments.refund.notification.errorTitle'),
                        message: response.message
                    });
                    return;
                }
                this.createNotificationSuccess({
                    title: this.$tc('scalapay-payments.refund.notification.successTitle'),
                    message: this.$tc('scalapay-payments.refund.notification.successMessage')
                });
                this.reloadEntityData();
            });


        }, createdComponent() {

            this.versionContext = Shopware.Context.api;

            this.systemConfigApiService.getValues('ScalapayPayments.config', null)
                .then(values => {
                    this.config = values;
                });

            this.reloadEntityData();
        }, reloadEntityData() {
            this.isLoading = true;

            return this.orderRepository.get(this.orderId, this.versionContext, this.orderCriteria).then((response) => {

                this.order = response;

                const totalTransactions = this.order.transactions.length;
                const lastTransactionIndex = totalTransactions - 1;
                if (totalTransactions <= 0 || !this.order.transactions[lastTransactionIndex].paymentMethodId) {
                    this.isRefundAllowed = false;
                }

                const paymentMethodId = this.order.transactions[lastTransactionIndex].paymentMethodId;
                if (paymentMethodId === undefined || paymentMethodId === null) {

                    this.isRefundAllowed = false;
                    this.isLoading = false
                    this.refundedAmount = 0;
                    this.maxRefundableAmount = this.order.amountTotal - 0;
                    this.isRefundDisabled = (this.order.amountTotal - this.refundedAmount === 0);

                    return Promise.resolve();
                }

                const paymentMethodRepository = this.repositoryFactory.create('payment_method');
                paymentMethodRepository.get(paymentMethodId, Context.api).then((paymentMethod) => {
                    if (paymentMethod.formattedHandlerIdentifier !== scalapayFormattedHandlerIdentifier) {
                        this.isRefundAllowed = false;
                        this.isLoading = false;
                        return;
                    }

                    if (this.order.transactions[lastTransactionIndex].customFields === undefined || this.order.transactions[lastTransactionIndex].customFields === null) {
                        this.isRefundAllowed = false;
                        this.isLoading = false;
                        return;
                    }

                    this.scalapayOrderToken = this.order.transactions[lastTransactionIndex].customFields.scalapay_order_token;

                    if (!this.scalapayOrderToken || this.config['ScalapayPayments.config.paymentScalapayStatus'] !== 'enabled') {
                        this.isRefundAllowed = false;
                        this.isLoading = false;
                        return;
                    }

                    this.scalapayApiService.getRefundData(this.order.id, this.scalapayOrderToken).then((response) => {

                        this.isRefundAllowed = response.isAllowed;
                        this.refundedAmount = this.order.amountTotal - response.refundedAmount;
                        this.maxRefundableAmount = response.refundedAmount;
                        this.isRefundDisabled = (this.order.amountTotal - response.refundedAmount === this.order.amountTotal);
                        this.isLoading = false;

                    }).catch(() => {
                        this.isRefundAllowed = false;
                        this.isLoading = false;
                    })
                },)

                return Promise.resolve();
            }).catch(() => {
                return Promise.reject();
            });
        },
    },

    computed: {
        orderRepository() {
            return this.repositoryFactory.create('order');
        }, orderCriteria() {
            const criteria = new Criteria(this.page, this.limit);

            criteria.addAssociation('currency');
            criteria.addAssociation('transactions');
            criteria.getAssociation('transactions').addSorting(Criteria.sort('createdAt'));

            return criteria;
        }, globalPluginConfig() {
            let config = this.$parent.$parent.$parent.actualConfigData;
            if (config) {
                return config.null;
            }
            return this.$parent.$parent.$parent.$parent.actualConfigData.null;
        }, actualPluginConfig() {
            let currentSalesChannelId = this.$parent.$parent.$parent.currentSalesChannelId;
            if (typeof currentSalesChannelId !== 'undefined') {
                return this.$parent.$parent.$parent.actualConfigData[currentSalesChannelId];
            }
            currentSalesChannelId = this.$parent.$parent.$parent.$parent.currentSalesChannelId;
            return this.$parent.$parent.$parent.$parent.actualConfigData[currentSalesChannelId];
        }
    },

    created() {
        this.createdComponent();
    }
});
