<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Service;

use Exception;
use Monolog\Level;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;

class LoggerService
{
    public const LOG_SOURCE = 'scalapay_payments';
    public const LOG_ENTRY_CHANNEL = 'scalapay_payments';
    public const LOG_ENTRY_KEY_ADDITIONAL_DATA = 'additionalData';
    public const LOG_ENTRY_KEY_CHANNEL = 'channel';
    public const LOG_ENTRY_KEY_CONTEXT = 'context';
    public const LOG_ENTRY_KEY_LEVEL = 'level';
    public const LOG_ENTRY_KEY_MESSAGE = 'message';
    public const LOG_ENTRY_KEY_SHOP_CONTEXT = 'shopContext';
    public const LOG_ENTRY_KEY_SOURCE = 'source';

    /**
     * @var EntityRepository
     */
    private EntityRepository $logEntryRepository;

    /**
     * @var SettingsService
     */
    private SettingsService $settingsService;

    /**
     * @param EntityRepository $logEntryRepository
     * @param SettingsService $settingsService
     */
    public function __construct(EntityRepository $logEntryRepository, SettingsService $settingsService)
    {
        $this->logEntryRepository = $logEntryRepository;
        $this->settingsService = $settingsService;
    }


    /**
     * @param $message
     * @param Context $context
     * @param Exception|null $exception
     * @param array|null $additionalData
     * @param int|Level $level
     * @return void
     */
    public function addEntry(
        $message,
        Context $context,
        ?Exception $exception = null,
        ?array $additionalData = null,
        int|Level $level = Level::Debug
    ): void {
        if (!$this->settingsService->getSetting('paymentScalapayLoggingEnabled')) {
            return;
        }

        if (!is_array($additionalData)) {
            $additionalData = [];
        }

        // Add exception to array
        if ($exception !== null) {
            $additionalData['error'] = [
                'message' => $exception->getMessage(),
                'trace' => $exception->getTrace(),
            ];
        }

        // Add data to the log entry
        $logEntry = [
            self::LOG_ENTRY_KEY_MESSAGE => mb_substr('Scalapay - ' . $message, 0, 255),
            self::LOG_ENTRY_KEY_LEVEL => is_a($level, Level::class) ? $level->value : $level,
            self::LOG_ENTRY_KEY_CHANNEL => mb_substr(self::LOG_ENTRY_CHANNEL, 0, 255),
            self::LOG_ENTRY_KEY_CONTEXT => [
                self::LOG_ENTRY_KEY_SOURCE => self::LOG_SOURCE,
                self::LOG_ENTRY_KEY_ADDITIONAL_DATA => $additionalData,
                self::LOG_ENTRY_KEY_SHOP_CONTEXT => $context?->getVars(),
            ],
        ];

        // Insert the log entry in the database
        $this->logEntryRepository->create([$logEntry], $context);
    }
}