<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Storefront;

use Scalapay\Scalapay\Handlers\ScalapayPaymentHandler;
use Scalapay\Scalapay\Helper\GatewayHelper;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Checkout\Payment\SalesChannel\AbstractPaymentMethodRoute;
use Shopware\Core\Checkout\Payment\SalesChannel\PaymentMethodRouteResponse;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Annotation\Route;

#[Route(defaults: ['_routeScope' => ['store-api']])]
class PaymentMethodRouteFiltered extends AbstractPaymentMethodRoute
{

    /**
     * @var GatewayHelper
     */
    private GatewayHelper $gatewayHelper;

    /**
     * @var AbstractPaymentMethodRoute
     */
    private AbstractPaymentMethodRoute $decorated;

    /**
     * @var EntityRepository
     */
    private EntityRepository $orderRepository;

    /**
     * @var RequestStack
     */
    private RequestStack $requestStack;

    /**
     * @param AbstractPaymentMethodRoute $decorated
     * @param GatewayHelper $gatewayHelper
     * @param EntityRepository $orderRepository
     * @param RequestStack $requestStack
     */
    public function __construct(AbstractPaymentMethodRoute $decorated, GatewayHelper $gatewayHelper, EntityRepository $orderRepository,
        RequestStack $requestStack
    )
    {
        $this->gatewayHelper = $gatewayHelper;
        $this->decorated = $decorated;
        $this->orderRepository = $orderRepository;
        $this->requestStack = $requestStack;
    }

    public function getDecorated(): AbstractPaymentMethodRoute
    {
        return $this->decorated;
    }

    /**
     * @param Request $request
     * @param SalesChannelContext $context
     * @param Criteria $criteria
     * @return PaymentMethodRouteResponse
     */
    public function load(Request $request, SalesChannelContext $context, Criteria $criteria): PaymentMethodRouteResponse
    {
        $response = $this->decorated->load($request, $context, $criteria);
        $order = $this->getCurrentOrder($request, $context);

        if ($this->gatewayHelper->isScalapayAvailable($context, null, $order)) {
            return $response;
        }

        foreach ($response->getPaymentMethods() as $paymentMethod) {
            if ($paymentMethod->getHandlerIdentifier() === ScalapayPaymentHandler::class) {
                $response->getPaymentMethods()->remove($paymentMethod->getId());
                break;
            }
        }

        return $response;
    }

    /**
     * @param Request $request
     * @param SalesChannelContext $context
     * @return ?OrderEntity
     */
    private function getCurrentOrder(Request $request, SalesChannelContext $context): ?OrderEntity
    {
        $currentRequest = $this->requestStack->getCurrentRequest();
        $orderId = $request->get('orderId') ?:
            ($request->attributes?->getAlnum('orderId') ?: $currentRequest?->attributes?->getAlnum('orderId'));

        if (empty($orderId) || !$context->getCustomer()) {
            return null;
        }

        $criteria = new Criteria([$orderId]);
        $criteria->addAssociation('deliveries.shippingOrderAddress.country');
        $criteria->addFilter(new EqualsFilter('order.salesChannelId', $context->getSalesChannel()->getId()));
        $criteria->addFilter(new EqualsFilter('order.orderCustomer.customerId', $context->getCustomer()->getId()));
        $criteria->addSorting(new FieldSorting('order.createdAt', FieldSorting::DESCENDING));

        return $this->orderRepository->search($criteria, $context->getContext())->first();
    }
}