<?php
/**
 * Scalapay
 *
 * @author Scalapay Plugin Integration Team
 *
 * Copyright © All rights reserved.
 * See LICENCE.md for license details.
 */

declare(strict_types=1);

namespace Scalapay\Scalapay\Storefront\Subscriber;

use Scalapay\Scalapay\PaymentMethods\PaymentMethodInterface;
use Scalapay\Scalapay\Service\SettingsService;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Storefront\Page\Checkout\Cart\CheckoutCartPageLoadedEvent;
use Shopware\Storefront\Page\Product\ProductPageLoadedEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;


class WidgetSubscriber implements EventSubscriberInterface
{
    /**
     * @var SettingsService
     */
    private SettingsService $settingsService;

    /**
     * @var EntityRepository
     */
    private EntityRepository $languageRepositoryInterface;

    /**
     * @var EntityRepository
     */
    private EntityRepository $localeRepositoryInterface;

    /**
     * @param SettingsService $settingsService
     * @param EntityRepository $languageRepositoryInterface
     * @param EntityRepository $localeRepositoryInterface
     */
    public function __construct(
        SettingsService $settingsService,
        EntityRepository $languageRepositoryInterface,
        EntityRepository $localeRepositoryInterface
    ) {
        $this->settingsService = $settingsService;
        $this->languageRepositoryInterface = $languageRepositoryInterface;
        $this->localeRepositoryInterface = $localeRepositoryInterface;
    }

    /**
     * {@inheritDoc}
     *
     * @return array
     */
    public static function getSubscribedEvents(): array
    {
        return [
            CheckoutCartPageLoadedEvent::class => 'addConfigsToPage',
            ProductPageLoadedEvent::class =>'addConfigsToPage',
        ];
    }

    /**
     * Assign Scalapay settings to page view for widget loading
     *
     * @param CheckoutCartPageLoadedEvent|ProductPageLoadedEvent $event
     * @return void
     */
    public function addConfigsToPage(ProductPageLoadedEvent|CheckoutCartPageLoadedEvent $event): void
    {
        $salesChannelContext = $event->getSalesChannelContext();

        $scalapaySettings = $this->settingsService->getSettings(
            $salesChannelContext->getSalesChannel()->getId()
        );

        if ($event instanceof CheckoutCartPageLoadedEvent && 'Shopware\Storefront\Page\Checkout\Cart\CheckoutCartPage' === get_class($event->getPage())) {
            $widgetData = [
                'type' => PaymentMethodInterface::CART_TYPE,
                'customPosition' => $this->sanitizeCssSelectors($scalapaySettings['paymentScalapayCartWidgetPosition']),
                'amountSelectors' => $this->sanitizeCssSelectors($scalapaySettings['paymentScalapayCartPriceSelector']),
                'currencyPosition' => $scalapaySettings['paymentScalapayCartCurrencyPosition'],
                'currencyDisplay' => $scalapaySettings['paymentScalapayCartCurrencyDisplay'],
                'amountSeparator' => $scalapaySettings['paymentScalapayCartAmountSeparator'],
                'hide' => (bool) $scalapaySettings['paymentScalapayCartHide'],
                'hideInstallments' => $scalapaySettings['paymentScalapayCartHidePrice'] ? 'true' : 'false',
                'hideLearnMore' => $scalapaySettings['paymentScalapayCartHideLearnMore'] ? 'true' : 'false',
                'darkMode' => $scalapaySettings['paymentScalapayCartDarkMode'],
                'alignment' => $scalapaySettings['paymentScalapayCartAlignment'],
            ];
        }

        if ($event instanceof ProductPageLoadedEvent && 'Shopware\Storefront\Page\Product\ProductPage' === get_class($event->getPage())) {
            $widgetData = [
                'type' => PaymentMethodInterface::PRODUCT_TYPE,
                'customPosition' => $this->sanitizeCssSelectors($scalapaySettings['paymentScalapayProductWidgetPosition']),
                'amountSelectors' => $this->sanitizeCssSelectors($scalapaySettings['paymentScalapayProductPriceSelector']),
                'currencyPosition' => $scalapaySettings['paymentScalapayProductCurrencyPosition'],
                'currencyDisplay' => $scalapaySettings['paymentScalapayProductCurrencyDisplay'],
                'amountSeparator' => $scalapaySettings['paymentScalapayProductAmountSeparator'],
                'hide' => (bool) $scalapaySettings['paymentScalapayProductHide'],
                'hideInstallments' => $scalapaySettings['paymentScalapayProductHidePrice'] ? 'true' : 'false',
                'hideLearnMore' => $scalapaySettings['paymentScalapayProductHideLearnMore'] ? 'true' : 'false',
                'darkMode' => $scalapaySettings['paymentScalapayProductDarkMode'],
                'alignment' => $scalapaySettings['paymentScalapayProductAlignment'],
            ];
        }

        $event->getPage()->addArrayExtension(
                'scalapayConfig', [
                    'enabled' => (int)('enabled' === $scalapaySettings['paymentScalapayStatus']),
                    'merchantToken' => $scalapaySettings['paymentScalapayMerchantToken'],
                    'mode' => $scalapaySettings['paymentScalapayMode'],
                    'min' => $scalapaySettings['paymentScalapayMinAmount'],
                    'max' => $scalapaySettings['paymentScalapayMaxAmount'],
                    'locale' => $this->getLocaleVariable($event),
                    'customCss' => $scalapaySettings['paymentScalapayCustomCss'] ?? '',
                    ...$widgetData ?? []
                ]

        );
    }

    /**
     * Return locale
     *
     * @param $args
     * @return string
     */
    private function getLocaleVariable($args): string
    {
        $fallbackLocale = 'en';
        $widgetAvailableLocales = [
            'en_US',
            'en_GB',
            'fr_FR',
            'it_IT',
            'de_DE',
            'de_AT',
            'de_CH',
            'es_ES',
            'ca_ES',
            'pt_PT',
            'fi_FI',
        ];

        $salesChannelContext = $args->getSalesChannelContext();
        $context = $salesChannelContext?->getContext();

        if (null === $context || null === ($languageId = $context->getLanguageId())) {
            return $fallbackLocale;
        }

        $languageCriteria = new Criteria();
        $languageCriteria->addFilter(new EqualsFilter('id', $languageId));
        $languages = $this->languageRepositoryInterface->search($languageCriteria, $context);

        $localeCriteria = new Criteria();
        $localeCriteria->addFilter(new EqualsFilter('id', $languages->first()?->getLocaleId()));

        $locales = $this->localeRepositoryInterface->search($localeCriteria, $context);
        $currentLocale = str_replace('-', '_', (string)$locales->first()?->getCode());

        if (empty($currentLocale) || !in_array($currentLocale, $widgetAvailableLocales, true)) {
            return $fallbackLocale;
        }

        return substr(explode('_', $currentLocale)[0], 0, 2) ?: $fallbackLocale;
    }

    /**
     * @param string $selectors
     * @return string
     */
    private function sanitizeCssSelectors(string $selectors): string
    {
        return str_replace('"', '', trim($selectors, '"\''));
    }

}